lib.addCommand('gymcreator', {
    help = 'Open gym creator debug menu',
    restricted = false,
    params = {},
}, function(source, args)
    TriggerClientEvent('gym:openCreator', source)
end)

---@param source Source
---@param data {model: string|number, coords: vector4, machine: string}[]
lib.callback.register('gym:saveCreatorData', function(source, data)
    if not data or #data == 0 then return false end

    local machineGroups = {}
    for _, item in pairs(data) do
        if not machineGroups[item.machine] then
            machineGroups[item.machine] = {}
        end
        table.insert(machineGroups[item.machine], item.coords)
    end

    local sumX, sumY, sumZ = 0, 0, 0
    local totalPoints = 0
    for _, item in pairs(data) do
        sumX = sumX + item.coords.x
        sumY = sumY + item.coords.y
        sumZ = sumZ + item.coords.z
        totalPoints = totalPoints + 1
    end

    local avgCoords = vec3(
        math.floor(sumX / totalPoints * 100) / 100,
        math.floor(sumY / totalPoints * 100) / 100,
        math.floor(sumZ / totalPoints * 100) / 100
    )

    local locationConfig = {
        coords = avgCoords,
        props = {}
    }

    for machine, coordsList in pairs(machineGroups) do
        locationConfig.props[machine] = {}
        for _, coords in pairs(coordsList) do
            table.insert(locationConfig.props[machine], vec4(
                math.floor(coords.x * 1000) / 1000,
                math.floor(coords.y * 1000) / 1000,
                math.floor(coords.z * 1000) / 1000,
                math.floor(coords.w * 1000) / 1000
            ))
        end
    end

    local configPath = GetResourcePath(cache.resource) .. '/config/config_c.lua'
    local file = io.open(configPath, 'r')
    if not file then
        lib.print.error('Failed to open config file')
        return false
    end

    local content = file:read('*all')
    file:close()

    local locationsStart = content:find('locations%s*=%s*{')
    if not locationsStart then
        lib.print.error('Failed to find locations table')
        return false
    end

    local newLocation = string.format([[
        { -- Location generated by %s at %s
            coords = vec3(%.2f, %.2f, %.2f),

            props = {
]], GetPlayerName(source), os.date("%Y-%m-%d %H:%M:%S"), avgCoords.x, avgCoords.y, avgCoords.z)

    for machine, coordsList in pairs(machineGroups) do
        newLocation = newLocation .. string.format('                %s = {\n', machine)
        for _, coords in pairs(coordsList) do
            newLocation = newLocation .. string.format('                    vec4(%.3f, %.3f, %.3f, %.3f),\n', 
                coords.x, coords.y, coords.z, coords.w)
        end
        newLocation = newLocation .. '                },\n'
    end

    newLocation = newLocation .. [[            }
        }
]]

    local propsStart = content:find('%s*},%s*props%s*=%s*{')
    if not propsStart then
        lib.print.error('Failed to find props table')
        return false
    end

    local beforeProps = content:sub(1, propsStart - 1)
    if not beforeProps:match('},%s*$') then
        beforeProps = beforeProps:gsub('}%s*$', '},\n')
    end

    local newContent = beforeProps .. newLocation .. content:sub(propsStart)

    file = io.open(configPath, 'w')
    if not file then
        lib.print.error('Failed to write config file')
        return false
    end

    file:write(newContent)
    file:close()

    local machineCount = 0
    for _ in pairs(machineGroups) do
        machineCount = machineCount + 1
    end

    lib.print.info(string.format('Successfully saved gym location with %d machines and %d total points', 
        machineCount, totalPoints))
    lib.print.debug(string.format('Location center coords: %.2f, %.2f, %.2f', avgCoords.x, avgCoords.y, avgCoords.z))

    return true
end)