# Exports API Documentation

This document describes all available exports for the gym system. The API has been updated with new, more flexible exports while maintaining backward compatibility with deprecated functions.

---

## Table of Contents
- [Deprecated Exports](#deprecated-exports)
- [New/Current Exports](#newcurrent-exports)
- [Usage Examples](#usage-examples)

---

## Deprecated Exports

These exports are still functional but will show warnings when used. They are maintained for backward compatibility but should be replaced with the new API.

### `SpawnGymEquipment`
**⚠️ DEPRECATED** - Use native location-based API instead

Spawns a single piece of gym equipment at a specific location.

**Syntax:**
```lua
local success = exports['prompt_anim_core']:SpawnGymEquipment(equipmentType, coords, heading, locationName, instanceName)
```

**Parameters:**
- `equipmentType` (string) - Type of equipment (e.g., "speedbag", "leg_press", "gymbike")
- `coords` (vector3) - Spawn coordinates
- `heading` (number) - Direction the equipment faces (0-360)
- `locationName` (string, optional) - Location identifier (auto-detected if omitted)
- `instanceName` (string, optional) - Unique identifier (auto-generated if omitted)

**Returns:**
- `success` (boolean) - True if equipment was spawned successfully

---

### `DespawnGymEquipment`
**⚠️ DEPRECATED** - Use native location-based API instead

Removes a previously spawned piece of equipment.

**Syntax:**
```lua
local success, error = exports['prompt_anim_core']:DespawnGymEquipment(instanceName)
```

**Parameters:**
- `instanceName` (string) - Unique identifier of the equipment to remove

**Returns:**
- `success` (boolean) - True if equipment was removed
- `error` (string, optional) - Error message if removal failed

---

### `GetEquipmentTypes`
**⚠️ DEPRECATED** - Use `GetAvailableEquipmentTypes` instead

Returns a list of all available equipment types.

**Syntax:**
```lua
local types = exports['prompt_anim_core']:GetEquipmentTypes()
```

**Returns:**
- `types` (table<number, string>) - Array of equipment type names

---

### `IsEquipmentBusy`
**⚠️ DEPRECATED** - Limited to custom spawned equipment only

Checks if a specific equipment instance is currently in use.

**Syntax:**
```lua
local isBusy = exports['prompt_anim_core']:IsEquipmentBusy(instanceName)
```

**Parameters:**
- `instanceName` (string) - Equipment instance identifier

**Returns:**
- `isBusy` (boolean) - True if equipment is being used

---

### `GetSpawnedEquipment`
**⚠️ DEPRECATED** - Use `GetLocationEquipment` or `GetAllLocations` instead

Returns all custom-spawned equipment data.

**Syntax:**
```lua
local equipment = exports['prompt_anim_core']:GetSpawnedEquipment()
```

**Returns:**
- `equipment` (table<string, CustomEquipmentData>) - Map of instance names to equipment data

---

## New/Current Exports

These are the recommended exports for managing gym locations and equipment.

### `CreateGymLocation`
Creates a new gym location with optional equipment.

**Syntax:**
```lua
local success, error = exports['prompt_anim_core']:CreateGymLocation(locationName, data)
```

**Parameters:**
- `locationName` (string) - Unique identifier for the location
- `data` (table) - Location configuration:
  - `coords` (vector3, required) - Center point of the gym location
  - `renderDistance` (number, optional) - Custom render distance for this location
  - `props` (table, optional) - Equipment to spawn at this location:
    - Key: Equipment type (string)
    - Value: Array of vector4 positions (x, y, z, heading)

**Returns:**
- `success` (boolean) - True if location was created
- `error` (string, optional) - Error message if creation failed

**Example:**
```lua
local success = exports['prompt_anim_core']:CreateGymLocation('my_gym', {
    coords = vec3(100.0, 200.0, 50.0),
    renderDistance = 75.0,
    props = {
        speedbag = {
            vec4(105.0, 205.0, 50.0, 90.0),
            vec4(105.0, 207.0, 50.0, 90.0)
        },
        gymbike = {
            vec4(110.0, 200.0, 50.0, 180.0)
        }
    }
})
```

---

### `RemoveGymLocation`
Removes an entire gym location and all its equipment.

**Syntax:**
```lua
local success = exports['prompt_anim_core']:RemoveGymLocation(locationName)
```

**Parameters:**
- `locationName` (string) - Location identifier to remove

**Returns:**
- `success` (boolean) - True if location was removed

---

### `AddEquipmentToLocation`
Adds equipment to an existing gym location.

**Syntax:**
```lua
local success, count = exports['prompt_anim_core']:AddEquipmentToLocation(locationName, equipmentType, coords)
```

**Parameters:**
- `locationName` (string) - Target location identifier
- `equipmentType` (string) - Type of equipment to add
- `coords` (vector4 or table<vector4>) - Single position or array of positions

**Returns:**
- `success` (boolean) - True if equipment was added
- `count` (number, optional) - Number of equipment pieces added

**Example:**
```lua
-- Add single equipment
local success = exports['prompt_anim_core']:AddEquipmentToLocation('vinewood', 'leg_press', vec4(100, 200, 50, 180))

-- Add multiple equipment at once
local success = exports['prompt_anim_core']:AddEquipmentToLocation('vinewood', 'speedbag', {
    vec4(100, 200, 50, 90),
    vec4(100, 202, 50, 90),
    vec4(100, 204, 50, 90)
})
```

---

### `RemoveEquipmentFromLocation`
Removes equipment from a gym location.

**Syntax:**
```lua
local success, count = exports['prompt_anim_core']:RemoveEquipmentFromLocation(locationName, equipmentType, coords, tolerance)
```

**Parameters:**
- `locationName` (string) - Target location identifier
- `equipmentType` (string) - Type of equipment to remove
- `coords` (vector4 or table<vector4> or nil) - Specific positions to remove, or nil to remove all
- `tolerance` (number, optional) - Distance tolerance for position matching (default: 0.1)

**Returns:**
- `success` (boolean) - True if equipment was removed
- `count` (number, optional) - Number of equipment pieces removed

**Example:**
```lua
-- Remove all leg presses from location
local success = exports['prompt_anim_core']:RemoveEquipmentFromLocation('vinewood', 'leg_press', nil)

-- Remove specific equipment by position
local success = exports['prompt_anim_core']:RemoveEquipmentFromLocation('vinewood', 'speedbag', vec4(100, 200, 50, 90), 0.5)
```

---

### `GetLocationEquipment`
Retrieves all equipment at a specific location.

**Syntax:**
```lua
local equipment = exports['prompt_anim_core']:GetLocationEquipment(locationName)
```

**Parameters:**
- `locationName` (string) - Location identifier

**Returns:**
- `equipment` (table<string, vector4[]> or nil) - Map of equipment types to position arrays, or nil if location doesn't exist

**Example:**
```lua
local equipment = exports['prompt_anim_core']:GetLocationEquipment('vinewood')
if equipment then
    for equipmentType, positions in pairs(equipment) do
        print(string.format("%s: %d pieces", equipmentType, #positions))
    end
end
```

---

### `GetAllLocations`
Returns all gym locations and their data.

**Syntax:**
```lua
local locations = exports['prompt_anim_core']:GetAllLocations()
```

**Returns:**
- `locations` (table<string, table>) - Map of location names to location data

**Example:**
```lua
local locations = exports['prompt_anim_core']:GetAllLocations()
for locationName, locationData in pairs(locations) do
    print(string.format("Location: %s at %s", locationName, locationData.coords))
end
```

---

### `GetAvailableEquipmentTypes`
Returns all available equipment types defined in the configuration.

**Syntax:**
```lua
local types = exports['prompt_anim_core']:GetAvailableEquipmentTypes()
```

**Returns:**
- `types` (string[]) - Array of equipment type names

**Example:**
```lua
local types = exports['prompt_anim_core']:GetAvailableEquipmentTypes()
-- Returns: {"speedbag", "leg_press", "gymbike", "gymlatpull", ...}
```

---

## Usage Examples

### Creating a Complete Gym

```lua
-- Create a new gym location with equipment
local success = exports['prompt_anim_core']:CreateGymLocation('sandy_gym', {
    coords = vec3(1850.0, 3700.0, 34.0),
    renderDistance = 100.0,
    props = {
        speedbag = {
            vec4(1855.0, 3705.0, 34.0, 90.0),
            vec4(1855.0, 3707.0, 34.0, 90.0)
        },
        leg_press = {
            vec4(1860.0, 3700.0, 34.0, 180.0)
        },
        gymbike = {
            vec4(1865.0, 3698.0, 34.0, 270.0),
            vec4(1865.0, 3700.0, 34.0, 270.0)
        }
    }
})

if success then
    print("Sandy Shores gym created successfully!")
end
```

### Dynamically Adding Equipment

```lua
-- Add more equipment to an existing location
exports['prompt_anim_core']:AddEquipmentToLocation('sandy_gym', 'gymlatpull', {
    vec4(1870.0, 3705.0, 34.0, 0.0),
    vec4(1872.0, 3705.0, 34.0, 0.0)
})
```

### Managing Equipment Lifecycle

```lua
-- When your resource starts, create a gym
AddEventHandler('onResourceStart', function(resourceName)
    if GetCurrentResourceName() ~= resourceName then return end
    
    exports['prompt_anim_core']:CreateGymLocation('my_custom_gym', {
        coords = vec3(500.0, 500.0, 100.0),
        props = {
            speedbag = { vec4(505.0, 505.0, 100.0, 90.0) }
        }
    })
end)

-- When your resource stops, clean up
AddEventHandler('onResourceStop', function(resourceName)
    if GetCurrentResourceName() ~= resourceName then return end
    
    -- The gym system will auto-cleanup resources created by your resource
    -- But you can manually remove if needed:
    exports['prompt_anim_core']:RemoveGymLocation('my_custom_gym')
end)
```

### Checking Available Equipment

```lua
-- Get all equipment types
local types = exports['prompt_anim_core']:GetAvailableEquipmentTypes()
print(string.format("Available equipment types: %d", #types))

-- Get equipment at a specific location
local equipment = exports['prompt_anim_core']:GetLocationEquipment('vinewood')
if equipment then
    for equipType, positions in pairs(equipment) do
        print(string.format("%s: %d units", equipType, #positions))
    end
end
```

---

## Equipment Types Reference

Available equipment types (as of current config):
- `bench` - Bench press
- `vin_chu` - Pull-up bar
- `leg_press` - Leg press machine
- `speedbag` - Wall-mounted speed bag
- `gymbike` - Stationary bike
- `gymlatpull` - Lat pulldown machine
- `gympullmachine1` - Pull machine type 1 (seated cable row)
- `gympullmachine2` - Pull machine type 2 (chest press)
- `gymrowpull` - Rowing pull machine
- `gymspeedbag` - Free-standing speed bag

---

## Notes

- All location and equipment management is automatically synchronized to all clients
- Equipment created by a resource is automatically cleaned up when that resource stops
- The new API is location-based, making it easier to manage entire gyms
- Deprecated exports will continue to work but show console warnings
- All coordinates should use vector3 for positions and vector4 for positions with heading
