local server = {}
local framework

if GetResourceState('qbx_core') == 'started' then
    framework = 'qbx'
elseif GetResourceState('qb-core') == 'started' then
    framework = 'qb'
    QBCore = exports['qb-core']:GetCoreObject()
elseif GetResourceState('es_extended') == 'started' then
    framework = 'esx'
    ESX = exports.es_extended:getSharedObject()
end

-- Permission check for starting an exercise
-- Called before a player begins using gym equipment
---@param source number Player server ID
---@param data {machine: string, index: number, locationName: string, coords: vector4, isCustomEquipment: boolean}
---@return boolean?, string? -- true if can use, error message otherwise
server.canDoExercise = function(source, data)
    return true
end

-- Permission check for accessing a machine
-- Called when a player interacts with gym equipment
---@param source number Player server ID
---@param data {machine: string, index: number, locationName: string, coords: vector4, isCustomEquipment: boolean}
---@return boolean?, string? -- true if can use, error message otherwise
server.canUseMachine = function(source, data)
    return true
end

-- Get player's name for display purposes
-- Used in boxing ring to show fighter names
-- Returns formatted name with first initial of last name
---@param source number Player server ID
---@return string Player's display name
server.getPlayerName = function(source) -- used in boxing ring
    local response

    if framework == 'qbx' or framework == 'qb' then
        local Player = framework == 'qbx' and exports.qbx_core:getPlayer(source) or QBCore.Functions.GetPlayer(source)
        if Player then
            response = Player.PlayerData.charinfo.firstname .. ' ' .. Player.PlayerData.charinfo.lastname:sub(1, 1) .. '.'
        end
    elseif framework == 'esx' then
        local Player = ESX.GetPlayerFromId(source)
        if Player then
            response = Player.get('firstName') .. ' ' .. Player.get('lastName'):sub(1, 1) .. '.'
        end
    end

    if not response then
        response = GetPlayerName(source)
    end

    return response
end

-- Update player's gym statistics (condition and strength)
-- Saves to player metadata and syncs to client
-- Stats are capped at 100 maximum
---@param source number Player server ID
---@param data {condition: number, strength: number} Stats to add
server.updateStats = function(source, data)
    if framework == 'qbx' then
        local current = exports.qbx_core:GetMetadata(source, 'gym_data') or {condition = 0, strength = 0}
        exports.qbx_core:SetMetadata(source, 'gym_data', {
            condition = math.min(100, current.condition + data.condition),
            strength = math.min(100, current.strength + data.strength),
        })
    elseif framework == 'qb' then
        local Player = QBCore.Functions.GetPlayer(source)
        if Player then
            local current = Player.Functions.GetMetaData('gym_data') or {condition = 0, strength = 0}
            current.condition = math.min(100, current.condition + data.condition)
            current.strength = math.min(100, current.strength + data.strength)
            Player.Functions.SetMetaData('gym_data', current)
        end
    elseif framework == 'esx' then
        local Player = ESX.GetPlayerFromId(source)
        if Player then
            local current = Player.getMeta('gym_data') or {condition = 0, strength = 0}
            current.condition = math.min(100, current.condition + data.condition)
            current.strength = math.min(100, current.strength + data.strength)
            Player.setMeta('gym_data', current)
        end
    end

    TriggerClientEvent('gym:updateGymData', source, data)
end

server.boxingRing = {
    ---@param fighter1 table
    ---@param fighter2 table
    onFightStart = function(fighter1, fighter2)
        if framework == 'qbx' then
            exports['qbx_medical']:Revive(fighter1.source)
            exports['qbx_medical']:Revive(fighter2.source)
        elseif framework == 'qb' then
            TriggerClientEvent('hospital:client:Revive', fighter1.source)
            TriggerClientEvent('hospital:client:Revive', fighter2.source)
        elseif framework == 'esx' then
            TriggerClientEvent('esx_ambulancejob:revive', fighter1.source)
            TriggerClientEvent('esx_ambulancejob:revive', fighter2.source)
        end
        -- Custom logic when a boxing match starts
    end,

    ---@param fighter1 table
    ---@param fighter2 table
    ---@param reason string
    onFightEnd = function(fighter1, fighter2, reason)
        if framework == 'qbx' then
            exports['qbx_medical']:Revive(fighter1.source)
            exports['qbx_medical']:Revive(fighter2.source)
        elseif framework == 'qb' then
            TriggerClientEvent('hospital:client:Revive', fighter1.source)
            TriggerClientEvent('hospital:client:Revive', fighter2.source)
        elseif framework == 'esx' then
            TriggerClientEvent('esx_ambulancejob:revive', fighter1.source)
            TriggerClientEvent('esx_ambulancejob:revive', fighter2.source)
        end
        -- Custom logic when a boxing match ends
    end,
}

-- ESX framework: Load player's gym stats when they join
AddEventHandler('esx:playerLoaded', function(src, xPlayer)
    local meta = xPlayer.getMeta('gym_data') or {condition = 0, strength = 0}
    TriggerClientEvent('gym:updateGymData', src, {
        condition = meta.condition,
        strength = meta.strength,
    })
end)

-- QBCore/QBX framework: Load player's gym stats when they join
RegisterNetEvent('QBCore:Server:OnPlayerLoaded', function()
    local src = source --[[@as number]]
    if framework == 'qbx' then
        local meta = exports.qbx_core:GetMetadata(src, 'gym_data') or {condition = 0, strength = 0}
        TriggerClientEvent('gym:updateGymData', src, {
            condition = meta.condition,
            strength = meta.strength,
        })
    elseif framework == 'qb' then
        local Player = QBCore.Functions.GetPlayer(src)
        if Player then
            local meta = Player.Functions.GetMetaData('gym_data') or {condition = 0, strength = 0}
            TriggerClientEvent('gym:updateGymData', src, {
                condition = meta.condition,
                strength = meta.strength,
            })
        end
    end
end)

return server