const { readdirSync } = require('fs')
const crypto = require('crypto')

const resourceName = GetCurrentResourceName()
const resourcePath = GetResourcePath(resourceName)

const customOrder = [
    'shared/default-config.lua',
    'shared/functions/functions.lua',
    'shared/functions/auto-detect.lua',
    'shared/functions/locales.lua',
    'server/frameworks/company-money/template.lua'
]

const getAllFilesInDirectory = (dir) => {
    return readdirSync(`${resourcePath}/${dir}`, { withFileTypes: true })
        .filter((file) => file.name.endsWith('.lua') || file.isDirectory())
        .sort((a, b) => {
            const aPath = `${dir}/${a.name}`
            const bPath = `${dir}/${b.name}`
            const aIndex = customOrder.indexOf(aPath)
            const bIndex = customOrder.indexOf(bPath)

            if (aIndex !== -1 && bIndex !== -1) {
                return aIndex - bIndex
            } else if (aIndex !== -1) {
                return -1
            } else if (bIndex !== -1) {
                return 1
            }

            return a.name.localeCompare(b.name)
        })
        .flatMap((dirent) => {
            if (dirent.isDirectory()) {
                return getAllFilesInDirectory(`${dir}/${dirent.name}`).map((subPath) => `${dirent.name}/${subPath}`)
            }

            return dirent.name
        })
}

const delay = (ms) => new Promise((resolve) => setTimeout(resolve, ms))

const rawSharedFiles = getAllFilesInDirectory('shared')
const rawServerFiles = getAllFilesInDirectory('server')
const rawClientFiles = getAllFilesInDirectory('client')

const sharedFiles = rawSharedFiles.map((file) => `"shared/${file}"`)
const serverFiles = rawServerFiles.map((file) => `"server/${file}"`)
const clientFiles = rawClientFiles.map((file) => `"client/${file}"`)

const allFiles = [...sharedFiles, ...serverFiles, ...clientFiles]

const hash = crypto.createHash('sha256').update(allFiles.join(',')).digest('hex')

async function generateLoadFile() {
    const watchdogScript =
        GetResourceState('watchdog') === 'started'
            ? 'watchdog'
            : GetResourceState('loaf_watchdog') === 'started'
            ? 'loaf_watchdog'
            : null

    if (watchdogScript) {
        await delay(2000)

        StopResource(watchdogScript)

        await delay(2000)
    }

    const initLuaFile = `local sharedFiles = {
    ${sharedFiles.join(',\n    ')}
}

local serverFiles = {
    ${serverFiles.join(',\n    ')}
}

local clientFiles = {
    ${clientFiles.join(',\n    ')}
}

local wrapperState = GetResourceState("${resourceName}")
local hasWrapper = wrapperState == "started" or wrapperState == "starting"
local loadFromResource = hasWrapper and "${resourceName}" or GetCurrentResourceName()
local fileNamePrefix = hasWrapper and "" or "${resourceName}/"

local function LoadFiles(files)
    for i = 1, #files do
        local fileName = fileNamePrefix .. files[i]

        Citizen.CreateThreadNow(function()
            local fileContent = LoadResourceFile(loadFromResource, fileName)

            if not fileContent then
                return
            end

            local loadFunction, errorMessage = load(fileContent, "@@${resourceName}/" .. fileName)

            if loadFunction then
                pcall(loadFunction)
            else
                print("^1[ERROR]^7: Failed to load file '" .. fileName .. "': " .. errorMessage)
            end
        end)
    end
end

LoadFiles(sharedFiles)

if IsDuplicityVersion() then
    LoadFiles(serverFiles)
else
    LoadFiles(clientFiles)
end
`

    SaveResourceFile(resourceName, 'load.lua', initLuaFile, -1)

    if (watchdogScript) {
        await delay(2000)

        StartResource(watchdogScript)
    }
}

if (GetResourceKvpString('hash') !== hash || !LoadResourceFile(resourceName, 'load.lua')) {
    console.log('Hash updated, or load.lua not found. Regenerating load.lua')

    SetResourceKvp('hash', hash)

    generateLoadFile().then(() => {
        console.log('load.lua regenerated.')
    })
}
