---@class LoafWrapperPointOptions
---@field coords vector3 | vector4
---@field radius number
---@field onEnter? fun(self: LoafWrapperPoint)
---@field onExit? fun(self: LoafWrapperPoint)
---@field inside? fun(self: LoafWrapperPoint)

---@class LoafWrapperPoint : LoafWrapperPointOptions
---@field distance number
---@field id number
---@field isInside boolean
---@field coords vector3
---@field insideInterval? Interval
---@field remove fun(self: LoafWrapperPoint)

---@type { [number]: LoafWrapperPoint }
local points = {}
local pointId = 0

local lastCoords = vector3(0, 0, 0)
local pointsChanged = false

local checkNearbyPointsInterval = Interval:new(function()
    local coords = GetEntityCoords(PlayerPedId())

    if #(coords - lastCoords) < 0.2 and not pointsChanged then
        return
    end

    lastCoords = coords
    pointsChanged = false

    ---@type LoafWrapperPoint[]
    local entered = {}
    ---@type LoafWrapperPoint[]
    local exited = {}

    for id, point in pairs(points) do
        local zDifference = math.abs(coords.z - point.coords.z)
        local distance = #(coords.xy - point.coords.xy)

        if zDifference < 2.0 and distance <= point.radius / 2 then
            point.distance = distance

            if not point.isInside then
                point.isInside = true

                entered[#entered+1] = point
            end
        elseif point.isInside then
            point.isInside = false

            exited[#exited+1] = point

            if point.insideInterval then
                point.insideInterval = point.insideInterval:remove()
            end
        end
    end

    -- onExit is triggered before onEnter, so points that e.g. clear help texts are triggered before new ones are shown

    for i = 1, #exited do
        local point = exited[i]

        if point.onExit then
            point:onExit()
        end
    end

    for i = 1, #entered do
        local point = entered[i]

        if point.onEnter then
            point:onEnter()
        end

        if point.inside and not point.insideInterval then
            point.insideInterval = Interval:new(function()
                if point and point.inside and point.isInside then
                    point:inside()
                end
            end)
        end
    end
end, 250, false)

---@param point LoafWrapperPoint
local function RemovePoint(point)
    if not point or not points[point.id] then
        return
    end

    if point.insideInterval then
        point.insideInterval = point.insideInterval:remove()
    end

    if point.isInside and point.onExit then
        point:onExit()
    end

    points[point.id] = nil
    pointsChanged = true
end

---@param options LoafWrapperPointOptions
---@return LoafWrapperPoint
function AddPoint(options)
    ---@cast options LoafWrapperPoint

    checkNearbyPointsInterval:toggle(true)

    pointId = pointId + 1
    pointsChanged = true

    options.id = pointId
    options.coords = options.coords.xyz
    options.isInside = false
    options.remove = RemovePoint

    points[options.id] = options

    return options
end
