---@class LoafWrapperMarkerDrawOptions
---@field type? number
---@field coords vector3 | vector4
---@field direction? vector3
---@field rotation? vector3
---@field radius? number
---@field height? number
---@field color? { r: number, g: number, b: number, a: number }
---@field bobUpAndDown? boolean
---@field faceCamera? boolean
---@field rotationOrder? number
---@field rotate? boolean
---@field texture? { dict?: string, name?: string }
---@field drawOnEntities? boolean

---@class LoafWrapperMarkerOptions : LoafWrapperMarkerDrawOptions
---@field onEnter? fun(self: LoafWrapperMarker)
---@field onExit? fun(self: LoafWrapperMarker)
---@field inside? fun(self: LoafWrapperMarker)
---@field visible? fun(self: LoafWrapperMarker) # This is called when the marker is visible
---@field onEnterViewDistance? fun(self: LoafWrapperMarker) # This is called when the marker is within the view distance, even if not visible
---@field onExitViewDistance? fun(self: LoafWrapperMarker) # This is called when the marker is no longer within the view distance, even if not visible
---@field drawDistance? number

---@class LoafWrapperMarker : LoafWrapperMarkerOptions
---@field remove fun(self: LoafWrapperMarker)
---@field toggleEnabled fun(self: LoafWrapperMarker, visible: boolean)
---@field isInside boolean
---@field drawDistance number

---@param marker LoafWrapperMarker
local function DrawMarkerInternal(marker)
	DrawMarker(
		marker.type,
		marker.coords.x,
		marker.coords.y,
		marker.coords.z,
		marker.direction.x,
		marker.direction.y,
		marker.direction.z,
		marker.rotation.x,
		marker.rotation.y,
		marker.rotation.z,
		marker.radius,
		marker.radius,
		marker.height,
		marker.color.r,
		marker.color.g,
		marker.color.b,
		marker.color.a,
		marker.bobUpAndDown,
		marker.faceCamera,
		marker.rotationOrder,
		marker.rotate,
		marker.texture.dict,
		marker.texture.name,
		marker.drawOnEntities
	)
end

---@param options LoafWrapperMarkerOptions
---@return LoafWrapperMarker, LoafWrapperPoint?, LoafWrapperPoint?
function AddMarker(options)
	---@cast options LoafWrapperMarker

	local enabled = false

	options.type = options.type or 1
	options.coords = options.coords.xyz
	options.direction = options.direction or vector3(0.0, 0.0, 0.0)
	options.rotation = options.rotation or vector3(0.0, 0.0, 0.0)
	options.radius = options.radius or 1.0
	options.height = options.height or 0.5
	options.color = options.color or (Config and Config.MarkerColor) or { r = 125, g = 75, b = 195, a = 150 }
	options.bobUpAndDown = options.bobUpAndDown == true
	options.faceCamera = options.faceCamera == true
	options.rotationOrder = options.rotationOrder or 2
	options.rotate = options.rotate == true
	options.texture = options.texture or {}
	options.drawOnEntities = options.drawOnEntities == true
	options.drawDistance = options.drawDistance or 100.0

	---@type number | nil
	local checkpoint

	---@type LoafWrapperPoint?
	local drawPoint = AddPoint({
		coords = options.coords,
		radius = options.drawDistance,
	})

	---@type LoafWrapperPoint?
	local markerPoint = AddPoint({
		coords = options.coords,
		radius = options.radius,
		onEnter = function()
			options.isInside = true
			print("Entered marker")

			if options.onEnter and enabled then
				options:onEnter()
			end
		end,
		onExit = function()
			options.isInside = false
			print("Exited marker")

			if options.onExit and enabled then
				options:onExit()
			end
		end,
		inside = function()
			if options.inside and enabled then
				options:inside()
			end
		end,
	})

	function drawPoint:onEnter()
		if options.onEnterViewDistance then
			options:onEnterViewDistance()
		end

		if enabled and options.color.a > 0 and options.type == 1 then
			if not checkpoint then
				checkpoint = CreateCheckpoint(
					options.type,
					options.coords.x,
					options.coords.y,
					options.coords.z,
					options.coords.x,
					options.coords.y,
					options.coords.z,
					options.radius,
					options.color.r,
					options.color.g,
					options.color.b,
					options.color.a,
					0
				)

				SetCheckpointCylinderHeight(checkpoint, options.height, options.height, options.radius)
				SetCheckpointIconScale(checkpoint, 0.0)
			end
		end
	end

	function drawPoint:onExit()
		if options.onExitViewDistance then
			options:onExitViewDistance()
		end

		if enabled and options.color.a > 0 and options.type == 1 and checkpoint then
			DeleteCheckpoint(checkpoint)

			checkpoint = nil
		end
	end

	function options:remove()
		if drawPoint then
			drawPoint:remove()
			drawPoint = nil
		end

		if markerPoint then
			markerPoint:remove()
			markerPoint = nil
		end
	end

	function options:toggleEnabled(newEnabled)
		newEnabled = newEnabled == true

		if enabled == newEnabled then
			return
		end

		enabled = newEnabled

		if markerPoint and markerPoint.isInside then
			if enabled and options.onEnter then
				options:onEnter()
			elseif not enabled and options.onExit then
				options:onExit()
			end
		end

		if enabled and options.color.a > 0 and options.type ~= 1 then
			function drawPoint:inside()
				DrawMarkerInternal(options)

				if options.visible then
					options:visible()
				end
			end
		elseif drawPoint and drawPoint.inside then
			drawPoint.inside = nil
		end
	end

	options:toggleEnabled(true)

	return options, markerPoint, drawPoint
end
