---@class LoafWrapperKeyBindOptions
---@field name string
---@field description string
---@field defaultKey string
---@field defaultMapper? string
---@field alternateKey? string
---@field alternateMapper? string
---@field onPress? fun(self: LoafWrapperKeyBind)
---@field onRelease? fun(self: LoafWrapperKeyBind, timePressed: integer)

---@class LoafWrapperKeyBind : LoafWrapperKeyBindOptions
---@field defaultMapper string
---@field pressed boolean
---@field startedPressing integer
---@field disabled boolean
---@field hash integer
---@field hex string
---@field instructional string

---@param options LoafWrapperKeyBindOptions
---@return LoafWrapperKeyBind
function AddKeyBind(options)
    ---@cast options LoafWrapperKeyBind

    assert(type(options) == "table", "options must be a table")
    assert(type(options.name) == "string", "options.name must be a string")
    assert(type(options.description) == "string", "options.description must be a string")

    options.defaultMapper = options.defaultMapper or "keyboard"

    local name = "loaf_" .. options.name:lower()
    local hash = joaat("+" .. name)

    options.hash = hash | 0x80000000
    options.hex = string.upper(string.format("%x", hash))

    if hash < 0 then
        options.hex = string.gsub(options.hex, string.rep("F", 8), "")
    end

    options.instructional = "~INPUT_" .. options.hex .. "~"

    RegisterCommand("+" .. name, function()
        if options.disabled or IsPauseMenuActive() then
            return
        end

        options.pressed = true
        options.startedPressing = GetGameTimer()

        TriggerEvent(GetCurrentResourceName() .. ":keyPressed", options)

        if options.onPress then
            options:onPress()
        end
    end, false)

    RegisterCommand("-" .. name, function()
        if options.disabled or IsPauseMenuActive() then
            return
        end

        local timePressed = GetGameTimer() - (options.startedPressing or 0)

        options.pressed = false
        options.startedPressing = nil

        TriggerEvent(GetCurrentResourceName() .. ":keyReleased", options)

        if options.onRelease then
            options:onRelease(timePressed)
        end
    end, false)

    RegisterKeyMapping("+" .. name, options.description, options.defaultMapper, options.defaultKey)

    if options.alternateKey then
        RegisterKeyMapping("~!+" .. name, options.description, options.alternateMapper or options.defaultMapper, options.alternateKey)
    end

    SetTimeout(500, function()
        TriggerEvent("chat:removeSuggestion", "/+%s" .. name)
        TriggerEvent("chat:removeSuggestion", "/-%s" .. name)
        TriggerEvent("chat:removeSuggestion", "/~!+%s" .. name)
    end)

    return options
end
