---@class LoafWrapperInteractOption
---@field label string
---@field icon? string # https://fontawesome.com/search?ic=free
---@field action fun(entity?: number)
---@field canInteract? fun(entity?: number): boolean
---@field [string] any

---@class LoafWrapperInteractPointOptions
---@field coords vector3 | vector4
---@field name string
---@field title? string # The menu title
---@field helpText string
---@field autoPerform? boolean # Automatically perform the action if there's only one option?
---@field color? { r: number, g: number, b: number, a: number }
---@field key? number | LoafWrapperKeyBind
---@field controllerKeyBind? LoafWrapperKeyBind
---@field instructionalKey? string
---@field options LoafWrapperInteractOption[]
---@field markerRadius? number
---@field interactWidth? number
---@field interactLength? number
---@field interactHeight? number
---@field canSee? fun(): boolean

---@class LoafWrapperInteractPoint : LoafWrapperInteractPointOptions
---@field remove fun(self: LoafWrapperInteractPoint)
---@field toggleVisible fun(self: LoafWrapperInteractPoint, visible: boolean)
---@field marker? LoafWrapperMarker
---@field targetZone? table

---@type LoafWrapperInteractPoint | nil
local currentInteractPoint

---@param options LoafWrapperInteractOption[]
---@return boolean
local function CanInteractWithAnyAction(options)
    for i = 1, #options do
        local option = options[i]

        if not option.canInteract or option.canInteract() then
            return true
        end
    end

    return false
end

---@param interactPoint LoafWrapperInteractPoint
local function GetInteractableOptions(interactPoint)
    ---@type LoafWrapperMenuElement[]
    local menuElements = {}
    local options = interactPoint.options

    for i = 1, #options do
        local option = options[i]

        if option.canInteract and not option.canInteract() then
            goto continue
        end

        menuElements[#menuElements + 1] = {
            label = option.label,
            icon = option.icon,
            canInteract = option.canInteract,
            action = function()
                option.action()
            end,
        }

        ::continue::
    end

    return menuElements
end

---@param interactPoint LoafWrapperInteractPoint
local function OpenInteractMenu(interactPoint)
    local menuElements = GetInteractableOptions(interactPoint)

    if #menuElements == 0 then
        debugprint("No interact options available for:", interactPoint.name)
        return
    end

    if interactPoint.autoPerform and #menuElements == 1 then
        Citizen.CreateThreadNow(function()
            menuElements[1]:action()
        end)

        return
    end

    OpenMenu(interactPoint.title or interactPoint.helpText, menuElements)
end

---@type { [string]: LoafWrapperInteractPoint }
local interactPoints = {}

---@param interactPoint LoafWrapperInteractPoint
local function RemoveInteractPoint(interactPoint)
    if interactPoint.targetZone then
        exports.qtarget:RemoveZone(interactPoint.targetZone)

        interactPoint.targetZone = nil
    elseif interactPoint.marker then
        interactPoint.marker:remove()
    end

    interactPoints[interactPoint.name] = nil
end

---@param options LoafWrapperInteractPoint
local function AddInteractTargetZone(options)
    function options:toggleVisible(visible)
        if not visible then
            if options.targetZone then
                exports.qtarget:RemoveZone(options.targetZone)
                options.targetZone = nil
            end

            return
        end

        if options.targetZone then
            return
        end

        options.targetZone = exports.qtarget:AddBoxZone(
            options.name,
            options.coords.xyz - vector3(0.0, 0.0, 0.25),
            options.interactWidth or 2.0,
            options.interactLength or options.interactWidth or 2.0,
            {
                name = options.name,
                heading = options.coords.w or 0.0,
                debugPoly = Config.DebugPoly,
                minZ = options.coords.z - (options.interactHeight or 2) / 2,
                maxZ = options.coords.z + (options.interactHeight or 2) / 2,
            },
            { options = options.options }
        )
    end

    if not options.canSee or options.canSee() then
        options:toggleVisible(true)
    end
end

---@param options LoafWrapperInteractPoint
local function AddInteractMarker(options)
    local marker = AddMarker({
        coords = options.coords,
        radius = options.markerRadius,
        color = options.color,
    })

    local canInteractInterval = Interval:new(function()
        if not CanInteractWithAnyAction(options.options) or (options.canSee and not options.canSee()) then
            marker:toggleEnabled(false)
        else
            marker:toggleEnabled(true)
        end
    end, 500, false)

    function marker:onEnterViewDistance()
        canInteractInterval:toggle(true)
    end

    function marker:onExitViewDistance()
        canInteractInterval:toggle(false)
    end

    options.key = options.key or 51

    local keyNumber = type(options.key) == "number" and options.key --[[@as number]]
        or options.key.hash

    function marker:onEnter()
        currentInteractPoint = options

        local helpText = options.helpText

        if options.autoPerform then
            local interactableOptions = GetInteractableOptions(options)

            if #interactableOptions == 1 then
                helpText = interactableOptions[1].label
            end
        end

        if not IsUsingKeyboard(0) and options.controllerKeyBind then
            DrawHelpText(helpText, options.controllerKeyBind.hash, options.controllerKeyBind.instructional)
        else
            DrawHelpText(
                helpText,
                keyNumber,
                type(options.key) == "table" and options.key.instructional or options.instructionalKey
            )
        end
    end

    if type(options.key) == "number" then
        function marker:inside()
            if IsControlJustReleased(0, keyNumber) then
                OpenInteractMenu(options)
            end
        end
    end

    function marker:onExit()
        currentInteractPoint = nil

        ClearHelpText()
        CloseMenu()
    end

    function options:toggleVisible(visible)
        marker:toggleEnabled(visible)
    end

    options.marker = marker
end

---@param options LoafWrapperInteractPointOptions
---@return LoafWrapperInteractPoint
function AddInteractPoint(options)
    ---@cast options LoafWrapperInteractPoint

    if Config.Target then
        AddInteractTargetZone(options)
    else
        AddInteractMarker(options)
    end

    options.remove = RemoveInteractPoint

    interactPoints[options.name] = options

    return options
end

---@param keyBind LoafWrapperKeyBind
AddEventHandler(GetCurrentResourceName() .. ":keyPressed", function(keyBind)
    if
        not currentInteractPoint
        or type(currentInteractPoint.key) ~= "table"
        or currentInteractPoint.key.name ~= keyBind.name
    then
        return
    end

    OpenInteractMenu(currentInteractPoint)
end)

AddEventHandler(GetCurrentResourceName() .. ":jobUpdated", function()
    for name, interactPoint in pairs(interactPoints) do
        if not interactPoint.canSee then
            goto continue
        end

        local canSee = interactPoint.canSee()

        interactPoint:toggleVisible(canSee)

        ::continue::
    end
end)
