---@param dict string
---@param timeout? number # Timeout in milliseconds. Defaults to 5000ms.
---@return string dict
---@return boolean success
function LoadDict(dict, timeout)
	timeout = timeout or 5000

	RequestAnimDict(dict)

	local startTime = GetGameTimer()

	while not HasAnimDictLoaded(dict) do
		Wait(0)

		if timeout and GetGameTimer() - startTime > timeout then
			debugprint("LoadDict: Timed out loading dict:", dict, "after", timeout, "ms")
			return dict, false
		end
	end

	return dict, true
end

---@param model string | number
---@return number
function GetModelHash(model)
	if type(model) == "string" then
		return joaat(model)
	end

	return model
end

---@param model string | number
---@param timeout? number # Timeout in milliseconds. Defaults to 5000ms.
---@return number? model
function LoadModel(model, timeout)
	model = GetModelHash(model)
	timeout = timeout or 5000

	RequestModel(model)

	local startTime = GetGameTimer()

	while not HasModelLoaded(model) do
		Wait(0)

		if GetGameTimer() - startTime > timeout then
			debugprint("LoadModel: Timed out loading model:", model, "after", timeout, "ms")
			return
		end
	end

	return model
end

---@param netId number
---@param timeout? number # Timeout in milliseconds. Defaults to 15000ms.
---@return number? entity
function WaitForNetId(netId, timeout)
	timeout = timeout or 15000

	local startTime = GetGameTimer()

	while not NetworkDoesNetworkIdExist(netId) or not NetworkGetEntityFromNetworkId(netId) do
		Wait(0)

		if timeout and GetGameTimer() - startTime > timeout then
			debugprint("WaitForNetId: Timed out waiting for netId:", netId, "after", timeout, "ms")
			return
		end
	end

	return NetworkGetEntityFromNetworkId(netId)
end

---Get the instructional key (\~INPUT_XXX\~) for a key mapping
---@param command string
---@return string
function GetInstructional(command)
	local hash = joaat("+" .. command)
	local hex = string.format("%x", hash):upper()

	if hash < 0 then
		hex = string.gsub(hex, string.rep("F", 8), "")
	end

	return "~INPUT_" .. hex .. "~"
end

---@class LoafWrapperNewBlipOptions
---@field label string
---@field coords vector2 | vector3 | vector4
---@field sprite? number
---@field color? number
---@field scale? number
---@field category? number
---@field shortRange? boolean # Defalts to true
---@field display? number # Defaults to 2 (map and minimap)

---@param options LoafWrapperNewBlipOptions
---@return number blip
function CreateBlip(options)
	local coords = options.coords
	local label = options.label
	local blip = AddBlipForCoord(coords.x, coords.y, coords.z or 0.0)

	if options.sprite then
		SetBlipSprite(blip, options.sprite)
	end

	if options.color then
		SetBlipColour(blip, options.color)
	end

	if options.scale then
		SetBlipScale(blip, options.scale)
	end

	if options.category then
		SetBlipCategory(blip, options.category)
	end

	SetBlipDisplay(blip, options.display or 2)

	if options.shortRange == nil or options.shortRange then
		SetBlipAsShortRange(blip, true)
	end

	BeginTextCommandSetBlipName("STRING")
	AddTextComponentString(label)
	EndTextCommandSetBlipName(blip)

	return blip
end

---@param desc string
---@param errType? "info" | "success" | "warning" | "error"
---@param title? string
function Notify(desc, errType, title)
	---@type string?
	local infoLevel = errType

	if Config.NotificationSystem == "ox_lib" then
		if errType == "info" then
			infoLevel = "inform"
		end

		lib.notify({
			title = title or desc,
			description = title and desc or nil,
			type = infoLevel,
		})
	elseif Config.NotificationSystem == "gta" then
		BeginTextCommandThefeedPost("STRING")
		AddTextComponentSubstringPlayerName(desc)
		EndTextCommandThefeedPostMessagetext(
			"CHAR_BLOCKED",
			"CHAR_BLOCKED",
			true,
			1,
			title or L("NOTIFICATION_TITLE"),
			""
		)
	elseif Config.Framework == "esx" then
		TriggerEvent("esx:showNotification", desc)
	elseif Config.Framework == "qb-core" or Config.Framework == "qbox" then
		TriggerEvent("QBCore:Notify", desc)
	end
end

local drawingHelpText = false

---@param text string
---@param control? number # The control key, e.g. 51
---@param instructionalKey? string # The instructional key, e.g. "\~INPUT_CONTEXT\~"
---@param key? string # The key to display, e.g. "E"
function DrawHelpText(text, control, instructionalKey, key)
	if drawingHelpText then
		debugprint("Already drawing help text")
		return
	end

	drawingHelpText = true

	local prefix = ""

	if control then
		prefix = "[" .. GetControlKey(control) .. "] "
	elseif key then
		prefix = "[" .. key .. "] "
	end

	if Config.HelpTextStyle == "ox_lib" then
		lib.showTextUI(prefix .. text)
	elseif Config.HelpTextStyle == "okokTextUI" then
		exports["okokTextUI"]:Open(prefix .. text, "darkgray", "right", true)
	elseif Config.HelpTextStyle == "jg-textui" then
		exports["jg-textui"]:DrawText(prefix .. text)
	elseif Config.HelpTextStyle == "cd_drawtextui" then
		TriggerEvent("cd_drawtextui:ShowUI", "show", prefix .. text)
	else
		if instructionalKey then
			prefix = instructionalKey .. " "
		end

		BeginTextCommandDisplayHelp("STRING")
		AddTextComponentSubstringPlayerName(prefix .. text)
		EndTextCommandDisplayHelp(0, true, true, 0)
	end
end

function ClearHelpText()
	drawingHelpText = false

	if Config.HelpTextStyle == "ox_lib" then
		lib.hideTextUI()
	elseif Config.HelpTextStyle == "okokTextUI" then
		exports["okokTextUI"]:Close()
	elseif Config.HelpTextStyle == "jg-textui" then
		exports["jg-textui"]:HideText()
	elseif Config.HelpTextStyle == "cd_drawtextui" then
		TriggerEvent("cd_drawtextui:HideUI")
	else
		ClearHelp(true)
	end
end

---Check if a ped is wearing heels
---@param ped? number # The ped to check. If not provided, it will default to the player's ped.
---@return boolean wearingHeels
function IsPedWearingHeels(ped)
	ped = ped or PlayerPedId()

	if GetEntityModel(ped) ~= `mp_f_freemode_01` then
		debugprint("Not female ped, not wearing heels")
		return false
	end

	local shoes = GetPedDrawableVariation(ped, 6)
	local componentHash = GetHashNameForComponent(ped, 6, shoes, GetPedTextureVariation(ped, 6))

	if componentHash == 0 then
		return shoes == 0 or (shoes >= 6 and shoes <= 8) or shoes == 12 or shoes == 14
	end

	return DoesShopPedApparelHaveRestrictionTag(componentHash, `HIGH_HEELS`, 0)
end
